od.AQuA.D <- function (F, N, M, plus=TRUE, mon=TRUE, ort=TRUE, t.max=60) 
{
    # The AQuA method for computing D-efficient exact designs of experiments
    # F ... model matrix of the type n times m, n>=m
    # N ... the required size of the design, N>=m
    # M ... optimal approximate information matrix (non-normalised!)
    # plus ... do we use the "+" quadratic approximation?
    #      plus=FALSE works marginally better or worse, depending on the problem
    # mon ... do we control the monotonicity? 
    #      mon=FALSE is often somewhat better, but mon=TRUE is safer
    # ort ... do we pre-orthogonalize the model?
    #      ort=FALSE is sometimes slightly better, but sometimes much worse
    # t.max ... total time of execution (cca)
    
    # Note: Computing M is not a part of AQuA, because, among other reasons,
    # M is necessary to know for efficiencies also before using any other heuristic. 
    
    start <- as.numeric(proc.time()[3])
    info <- paste("Running od.D.AQuA", plus, mon, ort, "for cca", t.max, "seconds")
    info <- paste(info, " starting at ", Sys.time(), ".", sep = "")
    print(info, quote = FALSE)
    prof <- matrix(0, ncol=2, nrow=0)
    m <- dim(F)[2]; n <- dim(F)[1]; one.m <- rep(1, m); inf.n <- rep(Inf, n) 
    eps <- 1e-14 # Note that eps should be very small, e.g., 1e-14
    
    # For simplicity, in the code Phi means directly the determinant
    # However, we always display the positively homogeneous version of Phi
    # in the initial (non-orthogonalized) model
    
    const <- 1; Phi.app <- det(M); Phi.hom <- Phi.app^(1/m); Fo <- F
    
    if(ort) {
        print(paste(format(Sys.time(), "%T"), "Computing the model orthogonalization..."), quote = FALSE)
        eig <- eigen(M, symmetric=TRUE)
        M12 <- eig$vectors %*% diag(sqrt(eig$values)) %*% t(eig$vectors)
        Fo <- Fo %*% solve(M12); M <- diag(m)
        const <- Phi.app; Phi.app <- 1; Phi.hom <- 1 
    }
    
    print(paste(format(Sys.time(), "%T"), "Computing the quadratic approximation..."), quote = FALSE)
    res <- Create.hS(Fo, M, crit="D", qav=ifelse(plus, "+", "-"))
    h <- res$h; S <- res$S

    print(paste(format(Sys.time(), "%T"), "Computing the vector of squared norms..."), quote = FALSE)
    sn2 <- apply(S * S, 1, sum)
    
    # For efficiency and also for a fair comparison, we will
    # compute K and L in the same way as in the KL algorithm. 
    K <- max(c(10, min(ceiling(sqrt(c(N, n))))))
    L <- max(c(10, ceiling(sqrt(n))))
    print(paste(format(Sys.time(), "%T"), "Setting K=", K, "and L=", L), quote = FALSE)
    
    next.sec <- n.ex <- n.rest <-0
    Phi.best <- -Inf; finish.all <- FALSE
    
    print(paste(format(Sys.time(), "%T"), "Initiating multiple restarts..."), quote = FALSE)
    while (!finish.all) {
        n.rest <- n.rest + 1
        
        # For speed and for a fair comparison, both KL and AQuA are
        # initiated from a fully random design as follows. (We assume that the regressors
        # of a random m-touple of different design points are independent with P = almost 1)
        
        print(paste(format(Sys.time(), "%T"), "Constructing a random initial design..."), quote = FALSE)
        w <- rep(0, n); w[sample(1:n, m)] <- 1
        if(N>m) {
            for(ii in 1:(N-m)) {
                i.add <- sample(1:n, 1) 
                w[i.add] <- w[i.add] + 1
            }
        }
        
        # Initializations
        STw <- t(S) %*% w; q.fun <-  h - 2 * S %*% STw
        M <- t((w %*% t(one.m)) * Fo) %*% Fo; Phi.w <- det(M)
        if(Phi.w > Phi.best) {
            w.best <- w; Phi.best <- Phi.w
        }

        print(paste(format(Sys.time(), "%T"), "Starting quadratic ascent..."), quote = FALSE)
        finish.all <- finish <- as.numeric(proc.time()[3]) > start + t.max
        
        while (!finish) {
            
            tm <- as.numeric(proc.time()[3]) - start
            if (tm > next.sec) {
                crit.best <- (Phi.best)^(1/m)
                info <- paste(format(Sys.time(), "%T"), round(tm,1),
                              "sec. Best efficiency in this iteration / overall:", 
                              round((Phi.w)^(1/m)/Phi.hom, 6), "/",
                              round((Phi.best)^(1/m)/Phi.hom,6))
                print(info, quote = FALSE)
                next.sec <- ceiling(tm)
            }
            prof <- rbind(prof, c(as.numeric(proc.time()[3])-start, (const*Phi.best)^(1/m)))
            # Here K and L are used as in the KL procedure 
            # for speed and also for a fair comparison.
            supp <- (1:n)[w > 0.5]; Kq.fun <- inf.n; Kq.fun[supp] <- q.fun[supp]
            Kord <- order(Kq.fun); Kact <- min(c(length(supp), K)); Kind <- Kord[1:Kact]
            Lact <- min(c(n, L)); Lind <- order(q.fun)[(n - Lact + 1):n]
            one.Kact <- rep(1, Kact); one.Lact <- rep(1, Lact)
            
            SK <- S[Kind,]; SL <- S[Lind,] 
            Delta.tab <- one.Kact %*%t(h[Lind] - 2*SL%*%STw - sn2[Lind]) + 2*SK%*%t(SL) -
                (h[Kind] - 2*SK%*%STw + sn2[Kind])%*%t(one.Lact)
            max.ind <- which(Delta.tab >= max(Delta.tab)-eps, arr.ind = TRUE)
            
            nb <- dim(max.ind)[1]; ni <- ifelse(nb==1, 1, sample(1:nb, 1))
            iK <- Kind[max.ind[ni, 1]]; iL <- Lind[max.ind[ni, 2]]
            w.tmp <- w; w.tmp[iK] <- w.tmp[iK] - 1; w.tmp[iL] <- w.tmp[iL] + 1
            M.tmp <- M - Fo[iK,]%*%t(Fo[iK,]) + Fo[iL,]%*%t(Fo[iL,]) 
            Phi.tmp <- det(M.tmp); imp <- FALSE
            
            if(mon) {
                if (Phi.tmp > Phi.w) {
                    Phi.w <- Phi.tmp; M <- M.tmp
                    w[iK] <- w[iK] - 1; w[iL] <- w[iL] + 1
                    STw <- STw + S[iL,] - S[iK,]
                    q.fun <- q.fun - 2 * S %*% (S[iL,] - S[iK,])
                    n.ex <- n.ex + 1; imp <- TRUE
                    if (Phi.w > Phi.best) {
                        w.best <- w; Phi.best <- Phi.w
                    }
                } 
            } else {
                # Here we really need the eps>0, because sometimes theoretical max of Delta.tab
                # is 0, but numerically it is small positive, which leads to looping.
                if (Delta.tab[max.ind[1, 1], max.ind[1,2]] > eps) {
                    Phi.w <- Phi.tmp; M <- M.tmp
                    w[iK] <- w[iK] - 1; w[iL] <- w[iL] + 1
                    STw <- STw + S[iL,] - S[iK,]
                    q.fun <- q.fun - 2 * S %*% (S[iL,] - S[iK,]) 
                    n.ex <- n.ex + 1; imp <- TRUE
                    if (Phi.w > Phi.best) {
                        w.best <- w; Phi.best <- Phi.w
                    }
                }
            }
            
            if (as.numeric(proc.time()[3]) > start + t.max) 
                finish.all <- TRUE
            if (finish.all || !imp) 
                finish <- TRUE
        }
    }
    
    t.act <- round(as.numeric(proc.time()[3]) - start, 2)
    info <- paste("od.D.AQA", plus, mon, ort, " finished after", t.act, "seconds at", Sys.time())
    print(info, quote = FALSE)
    info <- paste("with", n.rest, "restarts and", n.ex, "exchanges.")
    print(info, quote = FALSE)
    
    list(w.best = w.best, Phi.best = (const*Phi.best)^(1/m), eff.best = (Phi.best)^(1/m)/Phi.hom, t.act = t.act, rest=n.rest, exch=n.ex, prof=prof)
}
