Create.hS <- function (F, M, crit="D", qav="+") 
{
    # Creates the vectors "h" and the matrix "S" for the quadratic approximation
    # F ... the model matrix of the type n times m
    # M ... the anchor matrix for the quadratic approximation
    # crit ... the criterion to be approximated ("D" or "A").
    # qav ... variant of the quadratic approximation ("+", "-")
    
    library(matrixcalc)
    
    m <- dim(F)[2]; n <-dim(F)[1]; s <- m*(m+1)/2
    Gm <- duplication.matrix(m); eps <- 1e-12
    M1 <- solve(M)
    
    # For the D criterion and the + approximation
    if(crit=="D") {
        vec.M1 <- vec(M1)
        tilde.h <- t(Gm) %*% vec.M1
        
        if(qav=="+") {
            tilde.Q <- 0.5 * t(Gm) %*% (-vec.M1 %*% t(vec.M1) / m + M1 %x% M1) %*% Gm
        } else if(qav=="-") {
            tilde.Q <- (1/6) * t(Gm) %*% (vec.M1 %*% t(vec.M1) / m + M1 %x% M1) %*% Gm
        } else {
            stop(paste("Approximation variant", qav, "not supported"))
        }
        
    } else if(crit=="A") {
        M2 <- M1 %*% M1
        vec.M2 <- vec(M2)
        tilde.h <- t(Gm) %*% vec.M2
        
        if(qav=="+") {
            tilde.Q <- t(Gm) %*% (-vec.M2 %*% t(vec.M2) / sum(diag(M1)) + M1 %x% M2) %*% Gm
        } else if(qav=="-") {
            tilde.Q <- (1/3) * t(Gm) %*% (M1 %x% M2) %*% Gm
        } else {
            stop(paste("Approximation variant", qav, "not supported"))
        }
        
    } else {
        stop(paste("Criterion", crit, "not supported"))
    }

    if(rcond(tilde.Q) > eps) {
        print(paste(format(Sys.time(), "%T"), "Computing the Cholesky decomposition of tilde.Q"), quote=FALSE)
        tilde.C <- t(chol(tilde.Q))
    } else {
        print(paste(format(Sys.time(), "%T"), "Computing the spectral decomposition of tilde.Q"), quote=FALSE)    
        eig <- eigen(tilde.Q, symmetric=TRUE)
        mx <- max(eig$values); ts <- sum(eig$values/mx > eps)
        tilde.C <- eig$vectors[,1:ts] %*% diag(sqrt(eig$values[1:ts]))
    }
    diff <- max(abs(tilde.Q - tilde.C%*%t(tilde.C)))
    print(paste(format(Sys.time(), "%T"), "Relative error in the CC^T decomposition:", diff/max(abs(tilde.Q))), quote=FALSE)
    
    print(paste(format(Sys.time(), "%T"), "Creating the matrix H"), quote=FALSE)
    H <- matrix(0, nrow=n, ncol=s); k <- 0
    for(i1 in 1:m) {
        for(i2 in i1:m) {
            k <- k + 1
            H[,k] <- F[,i1] * F[,i2]
        }
    }
    
    print(paste(format(Sys.time(), "%T"), "Matrix H is ready"), quote=FALSE)
    
    list(h = H %*% tilde.h, S = H %*% tilde.C)
}
