function M = dist00(DR, PI, eps, maxit)
%DIST00 Calculate steady state distribution for given random process.
%
%	M = DIST00(DR, PI, EPS, MAXIT) computes steady state distribution
%	for firs-order N - state markov process with transition matrix PI,
%	where each agent is making also its own decisions. They are
%	described by decision rule DR.
%
%	PI is sized N x N, DR is sized N x S, where S is a number
%	of control variable stages.
%
%	EPS is desired precision, MAXIT maximal allowed number of
%	iterations. If iteration process works longer that 10 seconds,
%	statistics output is written to STDOUT.

start_t = clock;

if nargin < 2
   error('Invalid number of input arguments');
end

if (nargin < 3)
   eps = 0.0001;
end

if (nargin < 4)
   maxit = 500;
end

% get dimensions
SDR = size(DR);
N = SDR(1);
S = SDR(2);

% create decision transition matrix S x S x N
T = zeros(S, S, N);
for i = 1:N
   for j = 1:S 
      T(j, DR(i,j), i) = 1;
   end
end

M = ones(N, S);
sumM = sum(sum(M)');
M = M ./ sumM;
last_t = start_t;

% iterative process
for i = 1:maxit
   
   MP = M;
   
   % step over random process
   M = PI * M;
   sumM = sum(sum(M)');
   M = M ./ sumM;   
   
   % step over decisions
   for j = 1:N
      M(j,:) = M(j,:) * T(:,:,j);
   end
   
   % check if precision is not reached
   DF = [(abs(M-MP)+eps)./(1+abs(MP))];
   check = max(max(DF)');   
   if check < (eps + eps * eps)
      break      
   end
   
   % check if to print statistics
   if etime(clock, last_t) > 10
      disp(sprintf('Elapsed time: %08.2f  Iteration: %d  Precision %7.5f', etime(clock, start_t), i, check));
      last_t = clock;
   end
   

end

return