function [DR, V] = iter01(U,PI,beta,eps,maxit)
%ITER00 Iterate Bellman's equation with uncertain profits.
%
%	[DR, V] = ITER00(U,PI,beta,eps,maxit) iterate stochastic Bellman's
%	equation, until desired precision (eps) or maximal count of
%	iterations (maxit) is reached.
%
%	U is a N x (M x N) utility matrix, where N is dimension of
%	discretization of control variable. M is a discretization dimension
%	of a state variable.
%
%	PI is a M x M transition matrix for markov process with M states.
%
%	This function returns M x N value function matrix V, each row
%	represents one state and decision rule M x N matrix of integers DR,
%	where each item represents optimal decisions for given state (row)
%	and control variable value (column).
%
%	If iterative process works longer, each ten seconds statistics
%	to STDOUT is printed.

% check input parameters

start_t = clock;

if (nargin < 3)
   error('Not enought input arguments.');
end

if (nargin < 4)
   eps = 0.001;
end

if (nargin < 5)
   maxit = 999;
end

SU = size(U);
N = SU(1);
M = fix(SU(2)/N);
if (M ~= (SU(2)/N))
   error('Invalid dimensions of utility matrix.');
end

if (size(PI) ~= [M M])
   error('Invalid transition matrix dimensions.');
end

   
% first iteration

[V, DR] = max(U);
last_t = start_t;

% iterative process
for i = 1 : maxit
   
   % calculate discounted expected value function
   VP = V;
   VN = beta .* (reshape(V,N,M));
   NV = zeros(N,M * N);
   for j = 1:M
      NV(:,(((j - 1) * N) + 1):(j * N)) = repmat(VN(:,j),1,N);
   end
   
   % maximalize
   [V, DR] = max(U + NV);
   V = reshape((PI * (reshape(V,N,M)'))',1,N*M);
   
   % check if precision is not reached
   DF = [(abs(V-VP)+eps)./(1+abs(VP))]';
   check = max(DF);   
   if check < (eps + eps * eps);
      break      
   end   
   
   % check if to print statistics
   if etime(clock, last_t) > 10
      disp(sprintf('Elapsed time: %08.2f  Iteration: %d  Precision %7.5f', etime(clock, start_t), i, check));
      last_t = clock;
   end
   
end

% transformation of results
V = reshape(V,N,M)';
DR = reshape(DR,N,M)';

return