(* :Title: Option Valuation *)

(* :Author: Ross M. Miller *)

(* :Summary:
  Comprehensive set of option valuation analysis for Black-
  Scholes and binomial models.  Includes functions for
  sentivity analysis (delta, theta, etc.) and for
  constructing and evaluating financial decision trees
  including binomial trees with early exercise 
*)

(* :Context: OptionValue` *)

(* :Package Version: 1.0 *)

(* :Copyright: Copyright 1990  Addison-Wesley Publishing 
     Company. *)

(* :Keywords:
  Black-Scholes model, binomial model, decision trees,
  option valuation, financial analysis, investments
*)

(* Source:
  Ross M. Miller: "Option Valuation," in 
  Hal Varian, "Economic and Financial Modeling in
  Mathematica," 
*)

(* "Mathematica Versions 2.0 and 2.1 *)


BeginPackage["OptionValue`"]

RiskFreeRate::usage = "RiskFreeRate contains the continuously
compounded risk-free rate of return."

CallPayoff::usage = "CallPayoff[price,strike] gives the
intrinsic value of a call option at a given price for its
underlying stock."

PutPayoff::usage = "CallPayoff[price,strike] gives the
intrinsic value of a put option at a given price for its
underlying stock."

AuxBS::usage = "AuxBS[p,k,sd,r,t] is an auxiliary function,
which happens to equal delta, used in the Black-Scholes
model."

Norm::usage = "Norm[x] gives the cumulative normal distribution
function evaluated at a real-valued x"

BlackScholes::usage = "BlackScholes[p,k,sd,r,t] gives the
Black-Scholes value of a call option."

Delta::usage = "Delta[p,k,sd,r,t] gives the
Black-Scholes delta of a call option, its sensitivity to
changes in the price of the underlying stock."

Theta::usage = "Theta[p,k,sd,r,t] gives the
theta of a call option, the sensitivity of its value to changes
in the time until expiration."

Rho::usage = "Rho[p,k,sd,r,t] gives the rho of a
call option, the sensitivity of its value to changes in the
risk-free rate of return.  Rho returns a numerical value
when all of its arguments are numbers."

Kappa::usage = "Kappa[p,k,sd,r,t] gives the kappa
of a call option, the sensivity of its value to changes in
its volatility (sd).  (Kappa is also referred to as lambda
or vega.)."

OptionGamma::usage = "OptionGamma[p,k,sd,r,t] gives the
gamma of a call option, the sensitivity of its delta to changes
in the stock price."

OptionElasticity::usage = "OptionElasticity[p,k,sd,r,t] gives
the elasticity of a call option with respect to its stock
price."

ImpliedVolatility::usage = "ImpliedVolatility[p,k,r,t,optionprice]
gives the volatility of a call option that is implied by its
price."

ConsObj::usage = "ConsObj[obj,proplist] is a LISP-like object
constructor that attaches the properties in proplist to the object
symbol obj.  As in LISP, proplist alternatively gives the property
name and then its value."

AssetPrice::usage = "AssetPrice[option] gives the price of the
asset that underlies option."

AssetVolatility::usage = "AssetVolatility[option] gives the
volatility of the asset that underlies option."
ExerciseFunction::usage = "ExerciseFunction[option] gives the
exercise function for an option based on its Type property"

MakeAmerTree::usage = "MakeAmerTree[option,n] gives the binomial
financial decision tree for option during the timeto expiration
into n periods.  MakeAmerTree[option] assumes a default
value of four periods."

Successors::usage = "Successors[node[symbol,price,left,weight]]
gives a list of successor nodes to a node in an American binomial
option tree."

TreeValue::usage = "TreeValue[node[symbol,price,left,weight]]
gives the value of the financial decision tree with node as its
root."

Prob::usage = "Prob[node] gives the probability that a node will
be reached from its parent."

Expectation::usage = "Expectation[nodelist] gives the mathematical
expectation of the values of the nodes in nodelist."

Value::usage = "Value[option] gives the value of an option using
the options properties to determine the best valuation method."


Begin["`private`"]

RiskFreeRate = 0.04

CallPayoff[price_,strike_] = Max[0,price-strike]

PutPayoff[price_,strike_] = Max[0,strike-price]

AuxBS[p_,k_,sd_,r_,t_] = (Log[p/k]+r t)/
                        (sd Sqrt[t])+
                         .5 sd Sqrt[t]

BlackScholes[p_,k_,sd_,r_,t_] =
        p  Norm[AuxBS[p,k,sd,r,t]]- 
        k Exp[-r t] (Norm[AuxBS[p,k,sd,r,t]-sd Sqrt[t]])

Norm[z_?NumberQ]:= N[0.5 + 0.5 Erf[z/Sqrt[2]]]

Norm'[z_] = N[(1/Sqrt[2 Pi])] Exp[-z^2/2]

Delta[p_,k_,sd_,r_,t_] = D[BlackScholes[p,k,sd,r,t],p]

Theta[p_,k_,sd_,r_,t_] = -D[BlackScholes[p,k,sd,r,t],t]

Kappa[p_,k_,sd_,r_,t_] = D[BlackScholes[p,k,sd,r,t],sd]

Rho[p_,k_,sd_,r_,t_] = D[BlackScholes[p,k,sd,r,t],r]

OptionGamma[p_,k_,sd_,r_,t_] = D[Delta[p,k,sd,r,t],p]

OptionElasticity[p_,k_,sd_,r_,t_] =
                     p Delta[p,k,sd,r,t] / BlackScholes[p,k,sd,r,t]

ImpliedVolatility[p_,k_,r_,t_,optionprice_] :=
            sd /. FindRoot[BlackScholes[p,k,sd,r,t]==
                           optionprice,{sd,0.2}]

ConsObj[obj_,proplist_] := 
        Do[Block[{propname=proplist[[2*i-1]],
                  propval=proplist[[2*i]]},
                  propname[obj]^=propval],
           {i,Length[proplist]/2}]

AssetPrice[option_] := (AssetPrice[option] ^=
                        Price[Asset[option]])
                        
AssetVolatility[option_] := (AssetVolatility[option] ^=
                            Volatility[Asset[option]])
                            
ExerciseFunction[option_] := CallPayoff /;
                             Type[option]=="call"

ExerciseFunction[option_] := PutPayoff /;
                             Type[option]=="put"
RiskFreeRate = 0.04

MakeAmerTree[option_,n_:4] := 
            Block[{ChanceSymbol = Unique["cnode"],
            DecisionSymbol = Unique["dnode"],
            s = AssetPrice[option],
            k = ExercisePrice[option],
	        ex = ExerciseFunction[option],
	        t = ExpirationTime[option],
            sd= AssetVolatility[option] Sqrt[t/n] // N,
            a = Exp[RiskFreeRate t/n] // N,
            u = Exp[sd],
            d = 1/u},
            ConsObj[Evaluate[ChanceSymbol],
                 {Type,"chance",
                  Dfactor,1/a,
                  Upamt,u,
                  Downamt,d,
                  Upprob,(a-d)/(u-d) // N,
                  ExercisePrice,k,
                  ExerciseFunction,ex,
                  Succsym,DecisionSymbol}];
            ConsObj[Evaluate[DecisionSymbol],
                  {Type,"decision",
                   ExercisePrice,k,
                   ExerciseFunction,ex,
                   Succsym,ChanceSymbol}];
            node[ChanceSymbol,s,n,1]]

Successors[node[symbol_,price_,left_,weight_]] := 
{node[Succsym[symbol],
      Upamt[symbol]*price,left-1,
      Upprob[symbol]],
 node[Succsym[symbol],
      Downamt[symbol]*price,left-1,
      1-Upprob[symbol]]}  /;   Type[symbol]=="chance" ;
      
Successors[node[symbol_,price_,left_,weight_]] := 
{node[Succsym[symbol],
      price,
      left,
      1],
 node[Succsym[symbol],
      price,
      0,
      1]}                 /; Type[symbol]=="decision"

TreeValue[node[symbol_,price_,left_,weight_]] :=
(TreeValue[node[symbol,price,left,weight]]
  = Which[left==0,
      ExerciseFunction[symbol] [price,ExercisePrice[symbol]],
    Type[symbol]=="chance",
      Dfactor[symbol] *
       Expectation[Successors[node[symbol,price,left,weight]]],
    Type[symbol]=="decision",
      Max[TreeValue[Successors[node[symbol,price,left,weight]]]]]);

SetAttributes[TreeValue,Listable]
 
Prob[node_] := node[[4]] ; SetAttributes[Prob,Listable]  

Expectation[nodelist_] := Dot[Prob[nodelist], 
                              TreeValue[nodelist]]  
Value[option_] := BlackScholes[AssetPrice[option],
                               ExercisePrice[option],
                               AssetVolatility[option],
                               RiskFreeRate,
                               ExpirationTime[option]] /;
                           Type[option]=="call"
                           
Value[option_] := TreeValue[MakeAmerTree[option,8]] /;
                           Type[option]=="put"

End[]

EndPackage[]

(*


^*)